using System;
using System.IO;
using System.Xml;
using System.Threading;
using System.Collections;
using System.Diagnostics;
using System.Runtime.Remoting;

using Team_Project.Elements;
using Team_Project.Exceptions;
using Team_Project.PersistencyManagers;
using Team_Project.PersistencyManagers.Storages;

namespace Team_Project.PersistencyManagers.Protocols
{
	/// <summary>
	/// Protocollo per la comunicazione remota semplice. Questo protocollo
	/// utilizza remoting per la comunicazione con le altre copie.
	/// Comunica con le copie vicine (parent e figli diretti) e attende la propria
	/// risposta. Nel caso un'operazione vada in time-out il protocollo si occupa
	/// di segnalare al tree-manager l'irraggiungibilit della copia.
	/// <note>Nel prototipo le operazioni sulle copie vicine sono in serie, mentre sarebbe
	/// opportuno eseguirle in parallelo</note>
	/// </summary>
	public class DistribuitedSimpleProtocol : MarshalByRefObject, IDistribuitedPersistencyProtocol
	{
		/// <summary>
		/// Nome del progetto per il quale il protocollo  stato creato
		/// </summary>
		protected string project;
		/// <summary>
		/// Tipo di elemento di default per il quale l'istanza  stata creata
		/// </summary>
		protected Type elType;
		/// <summary>
		/// Nodo xml contenente la configurazione per questa istanza
		/// (individuata da progetto e tipo)
		/// </summary>
		protected XmlNode RemotingCfg;

		/// <summary>
		/// Costruisce un'istanza del protocollo
		/// </summary>
		/// <param name="proj">Progetto al quale si desidera accedere</param>
		/// <param name="ElementType">Tipo dell'elemento al quale si desidera accedere</param>
		/// <param name="cfg">Nodo xml contenente la configurazione</param>
		public DistribuitedSimpleProtocol(string proj,Type ElementType, XmlNode cfg)
		{
			Trace.WriteLine(Name + " creating for prj: " + proj);
			project = proj;
			RemotingCfg = cfg;
			elType = ElementType;
		}

#if DEBUG
		/// <summary>
		/// Funzione di debug per il test dell'eccezione DirectionNotSupported
		/// </summary>
		public void ThrowExc()
		{
			throw new DirectionNotSupportedException(this,Direction.Store,"Debug purpose only");
		}
#endif

		/// <summary>
		/// Accede al manager di protocolli remoto sulla copia specificata.
		/// L'oggetto restituito  un proxy per l'oggetto remoto e pu quindi
		/// generare un'eccezione alla prima invocazione di un metodo
		/// </summary>
		/// <param name="CopyName">Nome della copia di cui ottenere il protocol manager</param>
		/// <returns>Proxy verso l'istanza remota del protocol manager sulla copia richiesta
		/// da <paramref name="CopyName"/></returns>
#if DEBUG
		public
#else
		protected
#endif
		ProtocolManager RemoteManager(string CopyName)
		{
			string Port;
			string Protocol;
			string Resource;

			XmlNode copyNode = RemotingCfg.SelectSingleNode("RemotingConfig/RemoteCopy[@name=\"" + CopyName + "\"]");
			
			if(copyNode == null)
				copyNode = RemotingCfg.SelectSingleNode("RemotingConfig/Defaults");
			
			Port = ReadConfig(copyNode,"Port");
			Protocol = ReadConfig(copyNode,"Protocol");
			Resource = ReadConfig(copyNode,"Resource");

			string connection = string.Format("{0}://{1}:{2}/{3}",Protocol,MyDNS.Resolve(CopyName),Port,Resource);
			Trace.WriteLine(this.Name + " - RemoteManagerConnection: "+connection + " for " + CopyName);

			return (ProtocolManager)RemotingServices.Connect(typeof(ProtocolManager),connection);
		}

		private static string ReadConfig(XmlNode n, string key)
		{
			XmlNode keyN = n.SelectSingleNode(key);
			if(keyN != null)
			{
				XmlAttribute a = keyN.Attributes["value"];
				return keyN.Attributes["value"].Value;
			}
			else
				return n.SelectSingleNode("../Defaults/"+key).Attributes["value"].Value;
		}

		/// <summary>
		/// Attende il completamento dell'operazione, controllando che non si superi
		/// il time-out.
		/// Il time out utilizzato  definito nei 
		/// <see cref="Team_Project.Globals.TimeOut">parametri globali</see>
		/// </summary>
		/// <exception cref="Team_Project.Exceptions.TimeOutException">Lanciata nel caso
		/// l'operazione non sia completata entro il time-out</exception>
		/// <param name="operation">Descrizione dell'operazione da eseguire</param>
		/// <param name="aRes">Riferimento all'operazione asincrona di cui attendere
		/// il completamento</param>
		/// <param name="copyName">Nome della copia che si tenta di contattare</param>
		protected static void AwaitCompletition(string operation,IAsyncResult aRes,string copyName)
		{
			int waited = 0;
			while(waited <= Globals.Instance.TimeOut)
			{
				if(aRes.IsCompleted) break;
				waited += Globals.Instance.PollTime;
				Thread.Sleep(Globals.Instance.PollTime);
			}
			if(waited > Globals.Instance.TimeOut)
			{
				((CopiesTreeManager)Globals.Instance.Data["TreeManager"]).RemoveCopy(copyName);
				throw new TimeOutException(operation,copyName);
			}
		}

		#region IPersistencyProtocol Members

		/// <summary>
		/// Restituisce "DistribuitedSimpleProtocol"
		/// </summary>
		public virtual string Name
		{
			get{return "DistribuitedSimpleProtocol";}
		}

		/// <summary>
		/// Inizializza il protocollo.
		/// </summary>
		public virtual void Intit()
		{
			//Do nothing
		}

		/// <summary>
		/// Rilascia le risorse occupate
		/// </summary>
		public virtual void Close()
		{
			//Do nothing
		}

		/// <summary>
		/// Inserisce un elemento nello storage. Nel caso questo sia gi
		/// presente, l'elemento viene sovrascritto.
		/// L'elemento viene scritto solo in locale.
		/// </summary>
		/// <param name="el">Elemento da inserire nello storage</param>
		public virtual void Store(IProjectElement el)
		{
			Store(el,string.Empty);
		}

		/// <summary>
		/// Recupera un elemento dallo storage locale.
		/// </summary>
		/// <exception cref="Team_Project.Exceptions.ElementNotExistsException">
		/// L'elemento da recuperare non esiste
		/// </exception>
		/// <exception cref="Team_Project.Exceptions.LocationNotExistsException">
		/// La location indicata dal parametro location non esiste</exception>
		/// <param name="location">Percorso dell'elemento da recuperare</param>
		/// <param name="name">Nome dell'elemento da recuperare</param>
		/// <param name="elementType">Tipo dell'elemento da recuperare</param>
		/// <returns>Riferimento all'elemento recuperato dallo storage locale</returns>
		public virtual IProjectElement Retrive(string location, string name, Type elementType)
		{
			IStorage lclStorage = Globals.Instance.Storages.GetStorageFor(project,elementType);
			IProjectElement el = (IProjectElement)Activator.CreateInstance(elementType,new object[] {project,location,name});
			Globals.Instance.Mutexes.RequestRead(project,location + "||" + elementType.Name,name);
			try
			{el.LoadFrom(lclStorage);}
			finally
			{
				Globals.Instance.Mutexes.ReleaseRead(project,location + "||" + elementType.Name,name);
			}
			return el;
		}

		/// <summary>
		/// Controlla se una location esiste.
		/// </summary>
		/// <remarks>Viene controllato solo lo storage locale.</remarks>
		/// <param name="location">Percorso della location di cui controllare
		/// l'esistenza</param>
		/// <returns>true se la location esiste, false altrimenti</returns>
		public virtual bool QueryLocationExists(string location)
		{
			return Globals.Instance.Storages.GetStorageFor(project,elType).LocationExists(location);
		}

		/// <summary>
		/// Conrolla se un elemento esiste.
		/// </summary>
		/// <remarks>Viene controllato solo lo storage locale.</remarks>
		/// <param name="location">Percorso della location che si desidera controllare</param>
		/// <param name="name">Nome dell'elemento</param>
		/// <returns>true se l'elemento esiste, false atrimenti</returns>
		public virtual bool QueryElementExists(string location, string name)
		{
			IProjectElement el = (IProjectElement)Activator.CreateInstance(elType,new object[] {project,location,name});
			return Globals.Instance.Storages.GetStorageFor(project,elType).Exists(location,name+el.Suffix);
		}

		/// <summary>
		/// Crea una nuova location.
		/// </summary>
		/// <remarks>L'operazione ha successo anche in caso la location esista gi</remarks>
		/// <param name="location">Percorso della location da creare</param>
		public virtual void CreateLocation(string location)
		{
			CreateLocation(location,string.Empty);
		}

		/// <summary>
		/// Elimina una location.
		/// </summary>
		/// <remarks>La location viene distrutta in tutte le copie</remarks>
		/// <seealso cref="Team_Project.PersistencyManagers.Storages.IStorage.DestroyLocation">
		/// IStorage.DestroyLocation</seealso>
		/// <param name="location">Percorso della location da distruggere</param>
		/// <param name="checkEmpty">Se true, la funzione lancia un'eccezione
		/// se la location non esiste, se false il contenuto della location
		/// viene cancellato in ogni caso.</param>
		public virtual void DestroyLocation(string location, bool checkEmpty)
		{
			DestroyLocation(location,checkEmpty,string.Empty);
		}

		/// <summary>
		/// Distrugge un elemento.
		/// </summary>
		/// <seealso cref="Team_Project.PersistencyManagers.Storages.IStorage.Delete">
		/// IStorage.Delete</seealso>
		/// <remarks>L'elemento viene distrutto da tutte le copie</remarks>
		/// <param name="el">Elemento da distruggere</param>
		public virtual void DestroyElement(IProjectElement el)
		{
			DestroyElement(el,string.Empty);
		}

		/// <summary>
		/// Ottiene la lista completa dei nomi degli elementi in una certa location
		/// </summary>
		/// <param name="location">Location di cui controllare gli elementi</param>
		/// <returns>Un array contenente tutti i nomi degli elementi del tipo
		/// associato all'istanza del protocollo. Tali nomi sono privati dell'eventuale
		/// suffisso</returns>
		public virtual string[] QueryElemensInLocation(string location)
		{
			string [] res = Globals.Instance.Storages.GetStorageFor(project,elType).ElementsIn(location);
			ArrayList al = new ArrayList(res.Length);
			IProjectElement el = (IProjectElement)Activator.CreateInstance(elType,new object[] {project,location,"TEMP"});
			foreach(string s in res)
			{
				if(s.EndsWith(el.Suffix))
				{
					al.Add(s.Substring(0,s.Length-el.Suffix.Length));
				}
			}
			res = (string []) al.ToArray(typeof(string));
			return res;
		}

		/// <summary>
		/// Ottiene la lista delle sotto-location di una certa location
		/// </summary>
		/// <param name="baseL">Location della quale visualizzare le sotto location</param>
		/// <returns>Un array di stringhe contenente i nomi di tutte le sotto-location</returns>
		public virtual string[] QueryLocations(string baseL)
		{
			return Globals.Instance.Storages.GetStorageFor(project,elType).LocationsList(baseL);
		}

		#endregion

		#region IDistribuitedPersistencyProtocol Members
		/// <summary>
		/// Crea o aggiorna un elemento su tutte le copie. Questa funzione non dovrebbe
		/// essere richiamata in locale, viene richiamata da remoto per la gestione 
		/// dell'albero
		/// </summary>
		/// <param name="el">Elemento da scrivere sugli storage</param>
		/// <param name="source">Nome della location dalla quale proviene la richiesta.
		/// Se  la stringa vuota la richiesta viene considerata proveniente da locale</param>
		public virtual void Store(IProjectElement el,string source)
		{
			Trace.Write("Try to store " + el.FullName + "["+el.GetType().Name + "]");
			Trace.WriteLine(" from " + (source!=string.Empty?source:"local"));
			IStorage lclStorage = Globals.Instance.Storages.GetStorageFor(project,el.GetType());
			Globals.Instance.Mutexes.RequestWrite(project,el.Location+ "||" + el.GetType().Name,el.Name);
			try
			{
				el.WriteTo(lclStorage);
			}
			catch
			{
				if(lclStorage.Exists(el.Location,el.Name+el.Suffix))
					el.DestroyFrom(lclStorage);
			}
			finally
			{
				Globals.Instance.Mutexes.ReleaseWrite(project,el.Location+ "||" + el.GetType().Name,el.Name);}
			bool ok;
			do
			{
				ok = true;
				el = (IProjectElement)Activator.CreateInstance(el.GetType(),new object[]{project,el.Location,el.Name});
				Globals.Instance.Mutexes.RequestRead(project,el.Location+ "||" + el.GetType().Name,el.Name);
				try
				{
					el.LoadFrom(lclStorage);

					CopiesTreeManager ctm = (CopiesTreeManager)Globals.Instance.Data["TreeManager"];
					string parent = ctm.GetParent();
					string[] children = ctm.GetChildren();
					StoreDelegate stDel;
					GetProtocolDelegate gpDel;
					ProtocolManager rMng;
					IAsyncResult asRes;
			
					if(parent != source && parent != string.Empty)
					{
						rMng = RemoteManager(parent);
						gpDel = new GetProtocolDelegate(rMng.GetProtocolFor);
						asRes = gpDel.BeginInvoke(project,el.GetType().Assembly.FullName,el.GetType().FullName,Direction.Store,null,null);
						AwaitCompletition("Store - getMng",asRes,parent);
						try
						{
							IDistribuitedPersistencyProtocol pp =
								(IDistribuitedPersistencyProtocol) gpDel.EndInvoke(asRes);

							stDel = new StoreDelegate(pp.Store);
							asRes = stDel.BeginInvoke(el,Globals.Instance.LocalCopyName,null,null);
							AwaitCompletition("Store",asRes,parent);
							stDel.EndInvoke(asRes);
						}
						catch (System.Net.WebException)
						{
							ctm.RemoveCopy(parent);
						}
					}
					foreach(string s in children)
					{
						if(s != source)
						{
							rMng = RemoteManager(s);
							gpDel = new GetProtocolDelegate(rMng.GetProtocolFor);
							asRes = gpDel.BeginInvoke(project,el.GetType().Assembly.FullName,el.GetType().FullName,Direction.Store,null,null);
							AwaitCompletition("Store - getMng",asRes,s);
							try
							{
								IDistribuitedPersistencyProtocol pp = 
									(IDistribuitedPersistencyProtocol)gpDel.EndInvoke(asRes);
								stDel = new StoreDelegate(pp.Store);
								asRes = stDel.BeginInvoke(el,Globals.Instance.LocalCopyName,null,null);
								AwaitCompletition("Store",asRes,s);
								stDel.EndInvoke(asRes);
							}
							catch (System.Net.WebException)
							{
								ctm.RemoveCopy(s);
							}
						}
					}
				}
				catch(ConcurrencyException)
				{
					ok = false;
				}
				catch
				{
					if(lclStorage.Exists(el.Location,el.Name+el.Suffix))
						el.DestroyFrom(lclStorage);
				}
				finally
				{
					el.Dispose();
					Globals.Instance.Mutexes.ReleaseRead(project,el.Location+ "||" + el.GetType().Name,el.Name);
				}
			}while(!ok);
		}

		/// <summary>
		/// Crea una location su tutte le copie. Questa funzione non dovrebbe
		/// essere richiamata in locale, viene richiamata da remoto per la gestione 
		/// dell'albero
		/// </summary>
		/// <param name="location">Percorso della location da creare</param>
		/// <param name="source">Nome della location dalla quale proviene la richiesta.
		/// Se  la stringa vuota la richiesta viene considerata proveniente da locale</param>
		public virtual void CreateLocation(string location,string source)
		{
			IStorage lclStorage = Globals.Instance.Storages.GetStorageFor(project,elType);
			lclStorage.CreateLocation(location);
			CopiesTreeManager ctm = (CopiesTreeManager)Globals.Instance.Data["TreeManager"];
			string parent = ctm.GetParent();
			string[] children = ctm.GetChildren();
			CreateLocationDelegate clDel;
			GetProtocolDelegate gpDel;
			ProtocolManager rMng;
			IAsyncResult asRes;
			
			try
			{
				if(parent != source && parent != string.Empty)
				{
					rMng = RemoteManager(parent);
					gpDel = new GetProtocolDelegate(rMng.GetProtocolFor);
					asRes = gpDel.BeginInvoke(project,elType.Assembly.FullName,elType.FullName,Direction.Store,null,null);
					AwaitCompletition("CreateLocation - getMng",asRes,parent);
					try
					{
						IDistribuitedPersistencyProtocol pp =
							(IDistribuitedPersistencyProtocol) gpDel.EndInvoke(asRes);
						clDel = new CreateLocationDelegate(pp.CreateLocation);
						asRes = clDel.BeginInvoke(location,Globals.Instance.LocalCopyName,null,null);
						AwaitCompletition("CreateLocation",asRes,parent);
						clDel.EndInvoke(asRes);
					}
					catch (System.Net.WebException)
					{
						ctm.RemoveCopy(parent);
					}
				}
				foreach(string s in children)
				{
					if(s != source)
					{
						rMng = RemoteManager(s);
						gpDel = new GetProtocolDelegate(rMng.GetProtocolFor);
						asRes = gpDel.BeginInvoke(project,elType.Assembly.FullName,elType.FullName,Direction.Store,null,null);
						AwaitCompletition("Store - getMng",asRes,s);
						try
						{
							IDistribuitedPersistencyProtocol pp =
								(IDistribuitedPersistencyProtocol)gpDel.EndInvoke(asRes);
							/*IDistribuitedPersistencyProtocol pp =
								(IDistribuitedPersistencyProtocol)
								rMng.GetProtocolFor(project,elType.Assembly.FullName,elType.FullName,Direction.Store);*/
							clDel = new CreateLocationDelegate(pp.CreateLocation);
							asRes = clDel.BeginInvoke(location,Globals.Instance.LocalCopyName,null,null);
							AwaitCompletition("CreateLocation",asRes,s);
							clDel.EndInvoke(asRes);
						}
						catch (System.Net.WebException)
						{
							ctm.RemoveCopy(s);
						}
					}
				}
			}
			catch
			{
				if(lclStorage.LocationExists(location))
					lclStorage.DestroyLocation(location,false);
			}
		}
		
		/// <summary>
		/// Distrugge una location su tutte le copie. Questa funzione non dovrebbe
		/// essere richiamata in locale, viene richiamata da remoto per la gestione 
		/// dell'albero
		/// </summary>
		/// <param name="location">Percorso della location da distruggere</param>
		/// <param name="checkEmpty">Parametro da passare allo storage. Vedi
		/// <see cref="Team_Project.PersistencyManagers.Storages.IStorage.DestroyLocation">checjEmpty</see> per i dettagli.</param>
		/// <param name="source">Nome della location dalla quale proviene la richiesta.
		/// Se  la stringa vuota la richiesta viene considerata proveniente da locale</param>
		public virtual void DestroyLocation(string location, bool checkEmpty,string source)
		{
			IStorage lclStorage = Globals.Instance.Storages.GetStorageFor(project,elType);
			lclStorage.DestroyLocation(location,checkEmpty);
			CopiesTreeManager ctm = (CopiesTreeManager)Globals.Instance.Data["TreeManager"];
			string parent = ctm.GetParent();
			string[] children = ctm.GetChildren();
			DestroyLocationDelegate dlDel;
			IAsyncResult asRes;
			GetProtocolDelegate gpDel;
			ProtocolManager rMng;
			
			if(parent != source)
			{
				rMng = RemoteManager(parent);
				gpDel = new GetProtocolDelegate(rMng.GetProtocolFor);
				asRes = gpDel.BeginInvoke(project,elType.Assembly.FullName,elType.FullName,Direction.Store,null,null);
				AwaitCompletition("DestroyLocation - getMng",asRes,parent);
				try
				{
					IDistribuitedPersistencyProtocol pp =
						(IDistribuitedPersistencyProtocol) gpDel.EndInvoke(asRes);
					dlDel = new DestroyLocationDelegate(pp.DestroyLocation);
					asRes = dlDel.BeginInvoke(location,checkEmpty,Globals.Instance.LocalCopyName,null,null);
					AwaitCompletition("DestroyLocation",asRes,parent);
					dlDel.EndInvoke(asRes);
				}
				catch (System.Net.WebException)
				{
					ctm.RemoveCopy(parent);
				}
			}
			foreach(string s in children)
			{
				if(s != source)
				{
					rMng = RemoteManager(s);
					gpDel = new GetProtocolDelegate(rMng.GetProtocolFor);
					asRes = gpDel.BeginInvoke(project,elType.Assembly.FullName,elType.FullName,Direction.Store,null,null);
					AwaitCompletition("CreateLocation - getMng",asRes,parent);
					try
					{
						IDistribuitedPersistencyProtocol pp =
							(IDistribuitedPersistencyProtocol) gpDel.EndInvoke(asRes);
						dlDel = new DestroyLocationDelegate(pp.DestroyLocation);
						asRes = dlDel.BeginInvoke(location,checkEmpty,Globals.Instance.LocalCopyName,null,null);
						AwaitCompletition("DestroyLocation",asRes,s);
						dlDel.EndInvoke(asRes);
					}
					catch (System.Net.WebException)
					{
						ctm.RemoveCopy(s);
					}
				}
			}
		}

		/// <summary>
		/// Distrugge un elemento su tutte le copie. Questa funzione non dovrebbe
		/// essere richiamata in locale, viene richiamata da remoto per la gestione 
		/// dell'albero
		/// </summary>
		/// <param name="el">Elemento da distruggere</param>
		/// <param name="source">Nome della location dalla quale proviene la richiesta.
		/// Se  la stringa vuota la richiesta viene considerata proveniente da locale</param>
		public virtual void DestroyElement(IProjectElement el,string source)
		{
			IStorage lclStorage = Globals.Instance.Storages.GetStorageFor(project,el.GetType());
			Globals.Instance.Mutexes.RequestWrite(project,el.Location+ "||" + el.GetType().Name,el.Name);
			try{
				el.DestroyFrom(lclStorage);}
			finally{
				Globals.Instance.Mutexes.ReleaseWrite(project,el.Location+ "||" + el.GetType().Name,el.Name);}
			CopiesTreeManager ctm = (CopiesTreeManager)Globals.Instance.Data["TreeManager"];
			string parent = ctm.GetParent();
			string[] children = ctm.GetChildren();
			DestroyElementDelegate deDel;
			IAsyncResult asRes;
			GetProtocolDelegate gpDel;
			ProtocolManager rMng;
			
			if(parent != source)
			{
				rMng = RemoteManager(parent);
				gpDel = new GetProtocolDelegate(rMng.GetProtocolFor);
				asRes = gpDel.BeginInvoke(project,el.GetType().Assembly.FullName,el.GetType().FullName,Direction.Store,null,null);
				AwaitCompletition("DestroyElement - getMng",asRes,parent);
				try
				{
					IDistribuitedPersistencyProtocol pp =
						(IDistribuitedPersistencyProtocol) gpDel.EndInvoke(asRes);
					deDel = new DestroyElementDelegate(pp.DestroyElement);
					asRes = deDel.BeginInvoke(el,Globals.Instance.LocalCopyName,null,null);
					AwaitCompletition("DestroyElement",asRes,parent);
					deDel.EndInvoke(asRes);
				}
				catch (System.Net.WebException)
				{
					ctm.RemoveCopy(parent);
				}
			}
			foreach(string s in children)
			{
				if(s != source)
				{
					rMng = RemoteManager(s);
					gpDel = new GetProtocolDelegate(rMng.GetProtocolFor);
					asRes = gpDel.BeginInvoke(project,el.GetType().Assembly.FullName,el.GetType().FullName,Direction.Store,null,null);
					AwaitCompletition("DestroyElement - getMng",asRes,parent);
					try
					{
						IDistribuitedPersistencyProtocol pp =
							(IDistribuitedPersistencyProtocol) gpDel.EndInvoke(asRes);
						deDel = new DestroyElementDelegate(pp.DestroyElement);
						asRes = deDel.BeginInvoke(el,Globals.Instance.LocalCopyName,null,null);
						AwaitCompletition("DestroyElement",asRes,s);
						deDel.EndInvoke(asRes);
					}
					catch (System.Net.WebException)
					{
						ctm.RemoveCopy(s);
					}
				}
			}
		}
		#endregion
	}
}
